/* 
   Copyright (C) 1997,1998 by Didier Frick (dfrick@dial.eunet.ch)

   $Id: FadingImageSource.java,v 1.27 1999/07/13 16:26:59 dfrick Exp $

   This software may be freely modified and redistributed, provided this comment
   is left in the source. 
   
   If you make modifications or improvements to the code, you should add your
   name and e-mail address below, with a description of the changes.

   Name			E-mail			Description

   Didier Frick		dfrick@dial.eunet.ch	Initial release

*/

//package ch.dfr.diapo;

import java.awt.*;
import java.awt.image.*;
import java.util.*;

/**
 * This class implements an ImageProducer which produces interpolated images from a
 * series of images.
 */
public class FadingImageSource implements ImageProducer, ImageConsumer, Runnable {
    /**
     * The increment array (difference between two interpolated images)
     */
    protected byte incr[][];
    /**
     * The images to interpolate from
     */
    protected ImageProducer imageSource[];

    /**
     * The HSB converters
     * @see ch.dfr.diapo.HSBImageConsumer
     */
    protected HSBImageConsumer hsbMaker[]=new HSBImageConsumer[2];
    /**
     * The consumers for this ImageProducer
     */
    protected Vector consumers=new Vector();

    /**
     * Total number of images (including interpolated)
     */
    protected int total;
    /**
     * Number of interpolated images between each image
     */
    protected int between;
    /**
     * Time to display a non-interpolated image
     */
    protected int mainDelay;
    /**
     * Time to display an interpolated image
     */
    protected int delay;
    /**
     * Pixel width for the image
     */
    protected int width;
    /**
     * Pixel height for the image
     */
    protected int height;
    /**
     * Number of non-interpolated images
     */
    protected int nImages;
    /**
     * Index of the current HSBImageConsumer to add increment to
     */
    protected int currentHSB=1;
    /**
     * Number of the currently displayed non-interpolated image in the sequence
     */
    protected int imageNr=0;


    /**
     * Build a new FadingImageSource
     * @param b Number of interpolated images between each image
     * @param md Time to display a non-interpolated image (ms)
     * @param d Time to display an interpolated image (ms)
     * @param w  Pixel width for the image
     * @param h  Pixel height for the image
     */
    public FadingImageSource(Vector img, 
			     int b, 
			     int md,
			     int d,
			     int w,
			     int h) {
	between=b;
	mainDelay=md;
	delay=d;
	width=w;
	height=h;
	nImages=img.size();
	total=nImages*(between+1);
	int size=width*height;
	incr= new byte[size][3];
	imageSource=new ImageProducer[nImages];
	for(int j=0; j<nImages; j++) {
	    imageSource[j]=((Image)img.elementAt(j)).getSource();
	}
	for(int i=0; i<hsbMaker.length; i++) {
	    hsbMaker[i]= new HSBImageConsumer(width,height);
	}
    }

    /**
     * Build the increment array
     */
    protected void makeIncr(byte from[][], byte to[][]) {
	int div=between+1;
	try {
	    for(int i=0;; i++) {
		byte ft[]=from[i];
		byte tt[]=to[i];
		byte it[]=incr[i];
		it[0]=(byte)((tt[0]-ft[0])/div);
		it[1]=(byte)((tt[1]-ft[1])/div);
		it[2]=(byte)((tt[2]-ft[2])/div);
	    }
	} catch(ArrayIndexOutOfBoundsException x) {
	    ;
	}
    }

    /**
     * add the increment array to the destination
     */
    protected void addIncr(byte dest[][], byte incr[][]) {
	try {
	    for(int i=0;;i++) {
		byte dt[]=dest[i];
		byte it[]=incr[i];
		dt[0]+=it[0];
		dt[1]+=it[1];
		dt[2]+=it[2];
	    }
	} catch(ArrayIndexOutOfBoundsException x) {
	    ;
	}
    }

    /**
     * Send the new data to the consumers
     */
    protected void sendSource(int image, int cIndex) throws InterruptedException {
	ImageProducer currentSource=imageSource[image];
	ImageConsumer c=hsbMaker[cIndex];
	currentSource.removeConsumer(hsbMaker[1-cIndex]);
	synchronized(c) {
	    currentSource.startProduction(c);
	    c.wait();
	}
    }

    /**
     * Thread run method
     */
    public void run() {
	try {
	    if(imageNr==0)
		sendSource(0,0);
	    for(; !Thread.interrupted() ; imageNr++) {
		imageNr%=total;
		int d=delay;
		long start=System.currentTimeMillis();
		if((imageNr%(between+1)==0)) {
		    int mainImage=imageNr/(between+1);
		    imageSource[mainImage].startProduction(this);
		    d=mainDelay;
		    int nextHSB=currentHSB;
		    currentHSB=1-nextHSB;
		    int nextImage=(mainImage+1)%nImages;
		    sendSource(nextImage,nextHSB);
		    makeIncr(hsbMaker[currentHSB].hsbArray, hsbMaker[nextHSB].hsbArray);
		} else {
		    addIncr(hsbMaker[currentHSB].hsbArray, incr);
		    /* Implement fading & RGB image generation here */
		    hsbMaker[currentHSB].startProduction(this);
		}
		long rd=System.currentTimeMillis()-start;
		Thread.sleep((d>rd) ? (d-rd) : 50);
	    }
	} catch(Throwable x) {
	    ;
	}
    }

    // ImageProducer methods

    public void addConsumer(ImageConsumer c) {
	if(c!=null && !consumers.contains(c)) {
	    consumers.addElement(c);
	    c.setColorModel(ColorModel.getRGBdefault());
	    c.setDimensions(width,height);
	}
    }

    public boolean isConsumer(ImageConsumer c) {
	return consumers.contains(c);
    }

    public void removeConsumer(ImageConsumer c) {
	consumers.removeElement(c);
    }

    public void startProduction(ImageConsumer c) {
	addConsumer(c);
    }


    public void requestTopDownLeftRightResend(ImageConsumer c) {
	if(hsbMaker[currentHSB]!=null)
	    hsbMaker[currentHSB].requestTopDownLeftRightResend(this);
	else {
	    int mainImage=imageNr/(between+1);
	    imageSource[mainImage].requestTopDownLeftRightResend(this);
	}
    }

    public void setDimensions(int w, int h) {
	for(Enumeration e=consumers.elements(); e.hasMoreElements(); ) {
	    ImageConsumer c=(ImageConsumer)e.nextElement();
	    c.setDimensions(width,height);
	}
    }

    public void setProperties(Hashtable props) {
	for(Enumeration e=consumers.elements(); e.hasMoreElements(); ) {
	    ImageConsumer c=(ImageConsumer)e.nextElement();
	    c.setProperties(props);
	}
    }

    public void setColorModel(ColorModel cm) {
	for(Enumeration e=consumers.elements(); e.hasMoreElements(); ) {
	    ImageConsumer c=(ImageConsumer)e.nextElement();
	    c.setColorModel(cm);
	}
    }

    public void setHints(int hints) {
	hints&=~ImageConsumer.SINGLEFRAME;
	for(Enumeration e=consumers.elements(); e.hasMoreElements(); ) {
	    ImageConsumer c=(ImageConsumer)e.nextElement();
	    c.setHints(hints);
	}
    }

    public void setPixels(int x, int y, int w, int h, ColorModel cm ,
			  byte pixels[], int off, int scan) {
	for(Enumeration e=consumers.elements(); e.hasMoreElements(); ) {
	    ImageConsumer c=(ImageConsumer)e.nextElement();
	    c.setPixels(x,y,w,h,cm,pixels,off,scan);
	}
    }

    public void setPixels(int x, int y, int w, int h, ColorModel cm ,
			  int pixels[], int off, int scan) {
	for(Enumeration e=consumers.elements(); e.hasMoreElements(); ) {
	    ImageConsumer c=(ImageConsumer)e.nextElement();
	    c.setPixels(x,y,w,h,cm,pixels,off,scan);
	}
    }

    public void imageComplete(int status) {
	status=ImageConsumer.SINGLEFRAMEDONE;
	for(Enumeration e=consumers.elements(); e.hasMoreElements(); ) {
	    ImageConsumer c=(ImageConsumer)e.nextElement();
	    c.imageComplete(status);
	}
    }

}
