/* 
   Copyright (C) 1997,1998 by Didier Frick (dfrick@dial.eunet.ch)

   $Id: HSBImageConsumer.java,v 1.22 1999/06/16 17:09:51 dfrick Exp $

   This software may be freely modified and redistributed, provided this comment
   is left in the source. 
   
   If you make modifications or improvements to the code, you should add your
   name and e-mail address below, with a description of the changes.

   Name			E-mail			Description

   Didier Frick		dfrick@dial.eunet.ch	Initial release

*/

//package ch.dfr.diapo;

import java.awt.*;
import java.awt.image.*;
import java.util.*;

/**
 * Instances of this class are able to convert index color model and
 * RGB images into an HSB representation, and back into an RGB image.
 * Memory usage relative to image size: (3*image pixels) + (4*image width) bytes
 */

public class HSBImageConsumer implements ImageProducer, ImageConsumer {
    protected final static short BYTE_MAX=255;
    protected final static short BYTE_OFFSET=128;
    protected final static float div=(float)BYTE_MAX;

    protected byte hsbArray[][];
    protected byte hsbPalette[][]= new byte[256][3];
    protected int lineBuf[];
    protected int width;
    protected int height;
    protected float hsbs[]=new float[3];
    protected ColorModel cModel;
    protected ColorModel cm=ColorModel.getRGBdefault();

    /**
     * Build a new HSBImageConsumer with the given width and height.
     */
    public HSBImageConsumer(int w, int h) {
	width=w;
	height=h;
	hsbArray= new byte[w*h][3];
	lineBuf=new int[w];
    }

    public void setDimensions(int w, int h) {
    }

    public void setProperties(Hashtable props) {
    }

    public void setColorModel(ColorModel c) {
	cModel=c;
	if(c instanceof IndexColorModel) {
	    IndexColorModel cm=(IndexColorModel)c;
	    int size=cm.getMapSize();
	    for(int i=0; i<size; i++) {
		int col=cm.getRGB(i);
		Color.RGBtoHSB((col>>16)&0xff,(col>>8)&0xff,col&0xff,hsbs);
		hsbPalette[i][0]=(byte)(hsbs[0]*BYTE_MAX-BYTE_OFFSET);
		hsbPalette[i][1]=(byte)(hsbs[1]*BYTE_MAX-BYTE_OFFSET);
		hsbPalette[i][2]=(byte)(hsbs[2]*BYTE_MAX-BYTE_OFFSET);
	    }
	}
    }

    public void setHints(int hints) {
    }

    public void setPixels(int x, int y, int w, int h, ColorModel cm ,
			  byte pixels[], int off, int scan) {
	if(w>width)
	    w=width;
	if(h>height)
	    h=height;
	for(int dy=0;dy<h;dy++) {
	    int index=dy*scan+off+x;
	    int tmp=(y+dy)*width;
	    for(int dx=w; --dx>=0; tmp++ )
		{
		    short val=pixels[index++];
		    if(val<0)
			val+=256;
		    System.arraycopy(hsbPalette[val],0,hsbArray[tmp],0,3);
		}
	}
    }

    public void setPixels(int x, int y, int w, int h, ColorModel cm , int pixels[], int off, int scan){
	if(w>width)
	    w=width;
	if(h>height)
	    h=height;
	for(int dy=0;dy<h;dy++) {
	    int index=dy*scan+off+x;
	    int tmp=(y+dy)*width;
	    for(int dx=w; --dx>=0; tmp++ )
		{
		    int col=cm.getRGB(pixels[index++]);
		    Color.RGBtoHSB((col>>16)&0xff,(col>>8)&0xff,col&0xff,hsbs);
		    hsbArray[tmp][0]=(byte)(hsbs[0]*BYTE_MAX-BYTE_OFFSET);
		    hsbArray[tmp][1]=(byte)(hsbs[1]*BYTE_MAX-BYTE_OFFSET);
		    hsbArray[tmp][2]=(byte)(hsbs[2]*BYTE_MAX-BYTE_OFFSET);
		}
	}
    }

    public void imageComplete(int status) {
	synchronized(this) {
	    this.notifyAll();
	}
    }

    public void addConsumer(ImageConsumer c) {
    }

    public boolean isConsumer(ImageConsumer c) {
	return false;
    }

    public void removeConsumer(ImageConsumer c) {
    }

    public void startProduction(ImageConsumer c) {  
	c.setHints(TOPDOWNLEFTRIGHT | COMPLETESCANLINES | SINGLEPASS );
	for(int y=0; y<height; y++) {
	    for(int x=0; x<width; x++) {
		byte[] t=hsbArray[y*width+x];
		lineBuf[x]=Color.HSBtoRGB((t[0]+BYTE_OFFSET)/div,(t[1]+BYTE_OFFSET)/div,
					  (t[2]+BYTE_OFFSET)/div);
	    }
	    c.setPixels(0,y,width,1,cm,lineBuf,0,width);	    
	}
	c.imageComplete(ImageConsumer.SINGLEFRAMEDONE);
    }

    public void requestTopDownLeftRightResend(ImageConsumer c) {
	System.err.println("HSBI:RTDLRTR");
    }

}
